<?php
/* --------------------------------------------------------------
   UpdateTrackingCodesAction.php 2020-04-09
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Api\TrackingCode\Actions;

use Gambio\Admin\TrackingCode\Interfaces\TrackingCodeService;
use Gambio\Admin\TrackingCode\Model\TrackingCode;
use Gambio\Admin\TrackingCode\Model\TrackingCodes;
use Gambio\Api\TrackingCode\TrackingCodeApiRequestValidator;
use InvalidArgumentException;
use Slim\Http\Response;
use Slim\Http\ServerRequest as Request;

/**
 * Class UpdateTrackingCodesAction
 *
 * @package Gambio\Api\TrackingCode\Actions
 */
class UpdateTrackingCodesAction
{
    /**
     * @var TrackingCodeApiRequestValidator
     */
    private $requestValidator;
    
    /**
     * @var TrackingCodeService
     */
    private $service;
    
    
    /**
     * UpdateTrackingCodesAction constructor.
     *
     * @param TrackingCodeApiRequestValidator $requestValidator
     * @param TrackingCodeService             $service
     */
    public function __construct(
        TrackingCodeApiRequestValidator $requestValidator,
        TrackingCodeService $service
    ) {
        $this->requestValidator = $requestValidator;
        $this->service          = $service;
    }
    
    
    /**
     * @param Request  $request
     * @param Response $response
     * @param array    $args
     *
     * @return Response
     */
    public function __invoke(Request $request, Response $response, array $args): Response
    {
        $referencedErrors = $this->requestValidator->validatePutRequestBody($request->getParsedBody());
        if (count($referencedErrors) > 0) {
            return $response->withStatus(400)->withJson(['errors' => $referencedErrors]);
        }
        
        $trackingCodes = [];
        foreach ($request->getParsedBody() as $reference => $details) {
            try {
                $trackingCodes[$reference] = TrackingCode::createFromArray($details);
            } catch (InvalidArgumentException $exception) {
                $referencedErrors[$reference][] = $exception->getMessage();
            }
        }
        if (count($referencedErrors) > 0) {
            return $response->withStatus(422)->withJson(['errors' => $referencedErrors]);
        }
        
        $this->service->storeTrackingCodes(TrackingCodes::createWithReferences($trackingCodes));
        
        return $response->withStatus(204);
    }
}